import bpy
import os
import re
import shutil

from ..common import validate_name

class GB_OT_export2godot(bpy.types.Operator):
	"""Export whole collections to Godot throught glTF format"""
	bl_idname = "gb.export2godot"
	bl_label = "Export to Godot project"
	bl_options = {'REGISTER', 'UNDO'}

	def invoke(self, context, event):
		return self.execute(context)

	def execute(self, context):

		print("Root collection : {}".format(context.scene.root_collection))
		print("Godot path : {}".format(context.scene.godot_project_path))
		print("Blender repository path : {}".format(
			context.scene.blender_repository_path))



		def save_file_beforehand():
			self.report(
				{"WARNING"}, "File must be saved first - skipping export")
			bpy.ops.wm.save_mainfile('INVOKE_AREA')
			return {"CANCELLED"}

		def create_destination_folder():
			"""
			Define the destination folder and creates it if nonexistent.
			Return the path name
			"""

			final_path = bpy.data.filepath.replace(
				context.scene.blender_repository_path, context.scene.godot_project_path)
			gltf_path = os.path.splitext(final_path)[0]
			if not os.path.isdir(gltf_path):
				os.makedirs(gltf_path)
			return gltf_path

		def recurLayerCollection(layerColl, collName):
			""" Activate the selected collection for export"""
			found = None
			if (layerColl.name == collName):
				return layerColl
			for layer in layerColl.children:
				found = recurLayerCollection(layer, collName)
				if found:
					return found

		def export_content(scene_collection):
				
			def check_name(collection):
				is_valid = validate_name.validate_name(collection.name)
				if not is_valid:
					self.report({"WARNING"},
								"Name {} is not valid - skipping export".format(collection.name))
					return {"CANCELLED"}

			def export_to_file(tscn_collection):
				print("Exporting {}".format(tscn_collection.name))

				filename = tscn_collection.name
				# Export collection - parameters : https://docs.blender.org/api/current/bpy.ops.export_scene.html?highlight=export_scene#bpy.ops.export_scene.gltf
				print("Exporting to : {}/{}".format(gltf_path, filename))
				print("Exporting textures to : {}_{}_textures".format((os.path.splitext(os.path.basename(gltf_path)))[0], filename))
				bpy.ops.export_scene.gltf(
					filepath="{}/{}".format(gltf_path, filename),
					export_format="GLTF_SEPARATE",	# Export glTF Separate (.gltf + .bin + textures), Exports multiple files, with separate JSON, binary and texture data
					export_texture_dir="{}_{}_textures".format((os.path.splitext(os.path.basename(gltf_path)))[0], filename),	# Textures folder
					export_copyright=context.scene.licence,
					use_active_collection = True,
					use_renderable = True,
					export_cameras=False,
					export_lights=False,
					export_apply=True,					# Apply modifiers
					export_gn_mesh=True                 # Apply Geometry Nodes Instance
				)

			
			def get_godot_folder_content(directory, rootpath = context.scene.godot_project_path):
				godot_file_list = []
				for root, dirs, files in os.walk(rootpath):
					if os.path.basename(root) == directory:
						godot_files = os.listdir(root)
						for element in godot_files:
							if os.path.splitext(element)[1] == ".gltf":
								godot_file_list.append(os.path.splitext(element)[0])
				return godot_file_list

			
			def delete_godot_entry(element, rootpath = context.scene.godot_project_path):
				blend_filename = os.path.splitext(os.path.basename(bpy.data.filepath))[0]
				texture_folder = "{}_{}_textures".format(blend_filename, element)
				gltf_file = "{}.gltf".format(element)
				for root, dirs, files in os.walk(rootpath):
					if os.path.basename(root) == texture_folder:
						shutil.rmtree(root)
					if gltf_file in files:
						os.remove(os.path.join(root, gltf_file))

			
			def sync_deletions_in_godot():
				blend_path = bpy.data.filepath
				blend_filename = os.path.splitext(os.path.basename(blend_path))[0]
				godot_file_list = get_godot_folder_content(blend_filename)

				for element in godot_file_list:
					delete = True
					for collection in tscn_collections:
						if element == collection.name:
							delete = False
							break
					if delete:
						delete_godot_entry(element)

			tscn_collections = [ coll for coll in bpy.data.collections if scene_collection.user_of_id(coll)]

			for tscn in tscn_collections:
				print("---------------------------------------------------")
				check_name(tscn)
				# Activate proper collection for export
				layer_collection = bpy.context.view_layer.layer_collection
				layerColl = recurLayerCollection(layer_collection, tscn.name)
				if layerColl.exclude:
					print("{} is not activated - not exported".format(tscn.name))
				else:
					print("Set active collection to {}".format(tscn.name))
					bpy.context.view_layer.active_layer_collection = layerColl
					export_to_file(tscn)
			
			sync_deletions_in_godot()

		if not bpy.data.is_saved:
			save_file_beforehand()
		else:
			try:
				scn_col = bpy.data.collections[context.scene.root_collection]
			except KeyError:
				self.report({"WARNING"},
							"No \"{}\" root collection in the file - skipping export".format(context.scene.root_collection))
				return {"CANCELLED"}
			
			# Create the proper destination path
			gltf_path = create_destination_folder()

			export_content(scn_col)
			return {"FINISHED"}
